#ifndef __CTextStream__
#define __CTextStream__

//	===========================================================================

#include "../Basics/CString.hpp"
#include "../Basics/CSystemString.hpp"
#include "../Basics/CCountedObject.hpp"
#include <fstream>

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::Basics::CSystemString;
using std::ifstream;
using std::ofstream;

//	===========================================================================

namespace Exponent
{
	namespace IO
    {
		/**
		 * @class CTextStream CTextStream.hpp
		 * @brief Text stream to and from the disk
		 *
		 * Input and output from text files. Note that for input, no single line may be more than CTEXTSTREAM_MAX_CHARACTERS
		 *
		 * @date 17/08/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added read with in place string array
		 * @version 1.0.2 Added stream of counted objects
		 * @version 1.0.3 Added eof checking
		 * @version 1.0.4 Made it a CCountedObject
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CTextStream.hpp,v 1.6 2007/02/08 21:06:44 paul Exp $
		 */
		class CTextStream : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * @enum EStreamMode
			 * @brief Enumeration of stream mmodes
			 */
			enum EStreamMode
			{
				e_input = 0,		/**< Input mode */
				e_output,			/**< Output mode */
			};

//	===========================================================================

			const static long CTEXTSTREAM_MAX_CHARACTERS = 1024; /**< Max characters for an input stream */

//	===========================================================================

			/**
			 * Construction
			 * @param path The path to the file to read / write
			 * @param mode The streaming mode to use
			 */
			CTextStream(const CSystemString &path, const EStreamMode mode);

			/**
			 * Default construction
			 */
			CTextStream();

			/**
			 * Destruction
			 */
			virtual ~CTextStream();

//	===========================================================================

			/**
			 * Open the stream
			 * @param path The path to the file to read / write
			 * @param mode The streaming mode to use
			 */
			bool openStream(const CSystemString &path, const EStreamMode mode);

			/**
			 * Close the stream
			 */
			void closeStream();

			/**
			 * Is the stream open
			 * @retval bool True if stream open, false otherwise
			 */
			bool isStreamOpen() const { return m_streamIsOpen; }

//	===========================================================================

			/**
			 * Output operators
			 * @param string The string to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const char *string);

			/**
			 * Output operators
			 * @param string The string to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const CString &string);

			/**
			 * Output operators
			 * @param string The string to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const CSystemString &string);

			/**
			 * Output operators
			 * @param object The object to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const CCountedObject &object);

			/**
			 * Output operators
			 * @param value The value to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const long value);

			/**
			 * Output operators
			 * @param value The value to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const int value);

			/**
			 * Output operators
			 * @param value The value to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const double value);

			/**
			 * Output operators
			 * @param value The value to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const float value);

			/**
			 * Output operators
			 * @param value The value to write
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator << (const bool value);

			/**
			 * Input operators
			 * @param string The string to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (char *string);

			/**
			 * Input operators
			 * @param string The string to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (CString &string);

			/**
			 * Input operators
			 * @param string The string to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (CSystemString &string);

			/**
			 * Input operators
			 * @param value The value to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (long &value);

			/**
			 * Input operators
			 * @param value The value to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (int &value);

			/**
			 * Input operators
			 * @param value The value to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (double &value);

			/**
			 * Input operators
			 * @param value The value to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (float &value);

			/**
			 * Input operators
			 * @param value The value to read in to
			 * @retval CTextStream& A reference to this
			 */
			CTextStream &operator >> (bool &value);

//	===========================================================================

			/**
			 * Valid for output
			 * @retval bool True if file is open for output, false otherwise
			 */
			bool validForOutput();

			/**
			 * Valid for output
			 * @retval bool True if file is open for input, false otherwise
			 */
			bool validForInput();

			/**
			 * Has reached eof
			 * @retval bool True if reached EOF or not open, false otherwise
			 */
			bool hasReachedEOF();

			/**
			 * Flush to disk
			 */
			void flushToDisk();

//	===========================================================================

		protected:

//	===========================================================================

			bool m_streamIsOpen;								/**< Is the stream open */
			EStreamMode m_mode;									/**< Type of streaming */
			CSystemString m_filePath;							/**< the path to write to */
			ofstream m_outFile;									/**< the output file writer */
			ifstream m_inputFile;								/**< the input file reader */
			char m_buffer[CTEXTSTREAM_MAX_CHARACTERS];			/**< Reading buffer */
		};
	}
}
#endif	// End of CTextStream.hpp